document.addEventListener('DOMContentLoaded', async () => {
  const loginView = document.getElementById('loginView');
  const captureView = document.getElementById('captureView');
  const successView = document.getElementById('successView');
  const errorView = document.getElementById('errorView');
  
  const apiEndpointInput = document.getElementById('apiEndpoint');
  const loginEmailInput = document.getElementById('loginEmail');
  const loginPasswordInput = document.getElementById('loginPassword');
  const loginBtn = document.getElementById('loginBtn');
  
  const pageTitleEl = document.getElementById('pageTitle');
  const pageUrlEl = document.getElementById('pageUrl');
  const capsuleTitleInput = document.getElementById('capsuleTitle');
  const capsuleNotesInput = document.getElementById('capsuleNotes');
  const tagsContainer = document.getElementById('tagsContainer');
  const tagInput = document.getElementById('tagInput');
  const folderSelect = document.getElementById('folderSelect');
  const saveBtn = document.getElementById('saveBtn');
  const logoutBtn = document.getElementById('logoutBtn');
  
  const closeBtn = document.getElementById('closeBtn');
  const retryBtn = document.getElementById('retryBtn');
  const errorMessage = document.getElementById('errorMessage');
  
  let tags = [];
  let pageData = null;
  let sessionCookie = null;
  
  function showView(view) {
    [loginView, captureView, successView, errorView].forEach(v => v.classList.remove('active'));
    view.classList.add('active');
  }
  
  async function getStoredAuth() {
    return new Promise(resolve => {
      chrome.storage.local.get(['apiEndpoint', 'sessionCookie'], resolve);
    });
  }
  
  async function storeAuth(endpoint, cookie) {
    return new Promise(resolve => {
      chrome.storage.local.set({ apiEndpoint: endpoint, sessionCookie: cookie }, resolve);
    });
  }
  
  async function clearAuth() {
    return new Promise(resolve => {
      chrome.storage.local.remove(['apiEndpoint', 'sessionCookie'], resolve);
    });
  }
  
  async function getCurrentTab() {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    return tab;
  }
  
  async function getPageContent() {
    const tab = await getCurrentTab();
    try {
      const [result] = await chrome.scripting.executeScript({
        target: { tabId: tab.id },
        func: () => {
          const article = document.querySelector('article');
          const main = document.querySelector('main');
          const content = article || main || document.body;
          
          const cloned = content.cloneNode(true);
          cloned.querySelectorAll('script, style, nav, header, footer, aside, iframe').forEach(el => el.remove());
          
          return {
            title: document.title,
            url: window.location.href,
            content: cloned.innerText.trim().substring(0, 50000),
            description: document.querySelector('meta[name="description"]')?.content || ''
          };
        }
      });
      return result.result;
    } catch (e) {
      return {
        title: tab.title,
        url: tab.url,
        content: '',
        description: ''
      };
    }
  }
  
  async function fetchWithAuth(endpoint, path, options = {}) {
    const url = `${endpoint}${path}`;
    const response = await fetch(url, {
      ...options,
      credentials: 'include',
      headers: {
        'Content-Type': 'application/json',
        ...options.headers
      }
    });
    return response;
  }
  
  async function login(endpoint, email, password) {
    const csrfResponse = await fetch(`${endpoint}/api/csrf-token`, { credentials: 'include' });
    const { csrfToken } = await csrfResponse.json();
    
    const response = await fetch(`${endpoint}/api/auth/login`, {
      method: 'POST',
      credentials: 'include',
      headers: {
        'Content-Type': 'application/json',
        'X-CSRF-Token': csrfToken
      },
      body: JSON.stringify({ email, password })
    });
    
    if (!response.ok) {
      const data = await response.json();
      throw new Error(data.message || 'Login failed');
    }
    
    return response.json();
  }
  
  async function loadFolders(endpoint) {
    try {
      const response = await fetchWithAuth(endpoint, '/api/folders');
      if (response.ok) {
        const folders = await response.json();
        folderSelect.innerHTML = '<option value="">No folder</option>';
        folders.forEach(folder => {
          const option = document.createElement('option');
          option.value = folder.id;
          option.textContent = folder.name;
          folderSelect.appendChild(option);
        });
      }
    } catch (e) {
      console.error('Failed to load folders:', e);
    }
  }
  
  async function saveCapsule(endpoint, data) {
    const csrfResponse = await fetch(`${endpoint}/api/csrf-token`, { credentials: 'include' });
    const { csrfToken } = await csrfResponse.json();
    
    const response = await fetch(`${endpoint}/api/capsules`, {
      method: 'POST',
      credentials: 'include',
      headers: {
        'Content-Type': 'application/json',
        'X-CSRF-Token': csrfToken
      },
      body: JSON.stringify(data)
    });
    
    if (!response.ok) {
      const errorData = await response.json();
      throw new Error(errorData.message || 'Failed to save capsule');
    }
    
    return response.json();
  }
  
  function addTag(tag) {
    tag = tag.trim().toLowerCase();
    if (tag && !tags.includes(tag)) {
      tags.push(tag);
      renderTags();
    }
  }
  
  function removeTag(tag) {
    tags = tags.filter(t => t !== tag);
    renderTags();
  }
  
  function renderTags() {
    const existingTags = tagsContainer.querySelectorAll('.tag');
    existingTags.forEach(t => t.remove());
    
    tags.forEach(tag => {
      const tagEl = document.createElement('span');
      tagEl.className = 'tag';
      tagEl.innerHTML = `${tag} <span class="tag-remove" data-tag="${tag}">&times;</span>`;
      tagsContainer.insertBefore(tagEl, tagInput);
    });
  }
  
  tagsContainer.addEventListener('click', (e) => {
    if (e.target.classList.contains('tag-remove')) {
      removeTag(e.target.dataset.tag);
    }
  });
  
  tagInput.addEventListener('keydown', (e) => {
    if (e.key === 'Enter' || e.key === ',') {
      e.preventDefault();
      addTag(tagInput.value);
      tagInput.value = '';
    } else if (e.key === 'Backspace' && !tagInput.value && tags.length > 0) {
      removeTag(tags[tags.length - 1]);
    }
  });
  
  loginBtn.addEventListener('click', async () => {
    const endpoint = apiEndpointInput.value.trim().replace(/\/$/, '');
    const email = loginEmailInput.value.trim();
    const password = loginPasswordInput.value;
    
    if (!endpoint || !email || !password) {
      alert('Please fill in all fields');
      return;
    }
    
    loginBtn.disabled = true;
    loginBtn.innerHTML = '<span class="loading"><span class="spinner"></span> Connecting...</span>';
    
    try {
      await login(endpoint, email, password);
      await storeAuth(endpoint, 'authenticated');
      await initCaptureView(endpoint);
    } catch (e) {
      alert(e.message);
    } finally {
      loginBtn.disabled = false;
      loginBtn.textContent = 'Connect Account';
    }
  });
  
  saveBtn.addEventListener('click', async () => {
    const { apiEndpoint } = await getStoredAuth();
    const title = capsuleTitleInput.value.trim();
    
    if (!title) {
      alert('Please enter a title');
      return;
    }
    
    saveBtn.disabled = true;
    saveBtn.innerHTML = '<span class="loading"><span class="spinner"></span> Saving...</span>';
    
    try {
      const notes = capsuleNotesInput.value.trim();
      const sourceUrl = pageData.url;
      const content = notes 
        ? `${notes}\n\n---\n\nSource: ${sourceUrl}\n\n${pageData.content}`
        : `Source: ${sourceUrl}\n\n${pageData.content}`;
      
      await saveCapsule(apiEndpoint, {
        title,
        content,
        type: 'note',
        tags: tags.length > 0 ? tags : undefined,
        folderId: folderSelect.value || undefined
      });
      
      showView(successView);
    } catch (e) {
      errorMessage.textContent = e.message;
      showView(errorView);
    } finally {
      saveBtn.disabled = false;
      saveBtn.innerHTML = '<span class="btn-text">Save to Capsule</span>';
    }
  });
  
  logoutBtn.addEventListener('click', async () => {
    await clearAuth();
    showView(loginView);
  });
  
  closeBtn.addEventListener('click', () => {
    window.close();
  });
  
  retryBtn.addEventListener('click', async () => {
    const { apiEndpoint } = await getStoredAuth();
    if (apiEndpoint) {
      await initCaptureView(apiEndpoint);
    } else {
      showView(loginView);
    }
  });
  
  async function initCaptureView(endpoint) {
    showView(captureView);
    
    pageData = await getPageContent();
    pageTitleEl.textContent = pageData.title;
    pageUrlEl.textContent = pageData.url;
    capsuleTitleInput.value = pageData.title;
    
    await loadFolders(endpoint);
  }
  
  async function init() {
    const { apiEndpoint } = await getStoredAuth();
    
    if (apiEndpoint) {
      try {
        const response = await fetchWithAuth(apiEndpoint, '/api/auth/me');
        if (response.ok) {
          await initCaptureView(apiEndpoint);
          return;
        }
      } catch (e) {
        console.error('Auth check failed:', e);
      }
      await clearAuth();
    }
    
    showView(loginView);
  }
  
  init();
});
